<?php
/**
 * Copyright since 2007 PrestaShop SA and Contributors
 * PrestaShop is an International Registered Trademark & Property of PrestaShop SA
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.md.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/OSL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to https://devdocs.prestashop.com/ for more information.
 *
 * @author    PrestaShop SA and Contributors <contact@prestashop.com>
 * @copyright Since 2007 PrestaShop SA and Contributors
 * @license   https://opensource.org/licenses/OSL-3.0 Open Software License (OSL 3.0)
 */

namespace PrestaShop\PrestaShop\Adapter\Image\Uploader;

use ImageManager;
use ImageType;
use PrestaShop\PrestaShop\Core\Image\Exception\ImageOptimizationException;
use PrestaShop\PrestaShop\Core\Image\ImageFormatConfiguration;
use PrestaShop\PrestaShop\Core\Image\Uploader\Exception\ImageUploadException;
use PrestaShop\PrestaShop\Core\Image\Uploader\Exception\MemoryLimitException;
use PrestaShop\PrestaShop\Core\Image\Uploader\Exception\UploadedImageConstraintException;
use PrestaShop\PrestaShop\Core\Image\Uploader\ImageUploaderInterface;
use Symfony\Component\HttpFoundation\File\UploadedFile;

/**
 * Class CategoryThumbnailImageUploader.
 */
final class CategoryThumbnailImageUploader extends AbstractImageUploader implements ImageUploaderInterface
{
    public function __construct(
        private readonly ImageFormatConfiguration $imageFormatConfiguration,
    ) {
    }

    /**
     * {@inheritdoc}
     *
     * @throws MemoryLimitException
     * @throws ImageOptimizationException
     * @throws ImageUploadException
     * @throws UploadedImageConstraintException
     */
    public function upload($id, UploadedFile $uploadedImage)
    {
        $this->checkImageIsAllowedForUpload($uploadedImage);
        $this->deleteOldImage($id);
        $this->uploadImage($id, $uploadedImage);
        $this->generateDifferentTypes($id);
    }

    /**
     * Delete old category image.
     *
     * @param int $id
     */
    private function deleteOldImage($id)
    {
        if (file_exists(_PS_CAT_IMG_DIR_ . $id . '_thumb.jpg')) {
            unlink(_PS_CAT_IMG_DIR_ . $id . '_thumb.jpg');
        }
    }

    /**
     * @param int $id
     * @param UploadedFile $image
     *
     * @throws ImageOptimizationException
     * @throws ImageUploadException
     * @throws MemoryLimitException
     */
    private function uploadImage($id, UploadedFile $image)
    {
        $temporaryImageName = tempnam(_PS_TMP_IMG_DIR_, 'PS');
        if (!$temporaryImageName) {
            throw new ImageUploadException('Failed to create temporary category thumbnail image file');
        }
        // move_uploaded_file -  also checks that the given file is a file that was uploaded via the POST,
        // this prevents for example that a local file is moved
        if (!move_uploaded_file($image->getPathname(), $temporaryImageName)) {
            throw new ImageUploadException('Failed to upload category thumbnail image');
        }

        if (!ImageManager::checkImageMemoryLimit($temporaryImageName)) {
            throw new MemoryLimitException('Cannot upload category thumbnail image due to memory restrictions');
        }

        $optimizationSucceeded = ImageManager::resize(
            $temporaryImageName,
            _PS_IMG_DIR_ . 'c' . DIRECTORY_SEPARATOR . $id . '_thumb.jpg',
            null,
            null
        );

        if (!$optimizationSucceeded) {
            throw new ImageOptimizationException('Failed to optimize image after uploading');
        }

        unlink($temporaryImageName);
    }

    /**
     * @param int $id
     *
     * @throws ImageUploadException
     */
    private function generateDifferentTypes($id)
    {
        if (!file_exists(_PS_CAT_IMG_DIR_ . $id . '_thumb.jpg')) {
            return;
        }

        $configuredImageFormats = $this->imageFormatConfiguration->getGenerationFormats();
        $imagesTypes = ImageType::getImagesTypes('categories');
        foreach ($imagesTypes as $imageType) {
            foreach ($configuredImageFormats as $imageFormat) {
                $generated = ImageManager::resize(
                    _PS_CAT_IMG_DIR_ . $id . '_thumb.jpg',
                    _PS_CAT_IMG_DIR_ . $id . '_thumb-' . stripslashes($imageType['name']) . '.' . $imageFormat,
                    (int) $imageType['width'],
                    (int) $imageType['height'],
                    $imageFormat
                );

                if (!$generated) {
                    throw new ImageUploadException('Error occurred when uploading category thumbnail image');
                }
            }
        }
    }
}
